#!/usr/bin/env python
'''
@file device_io_entry_base.py

@brief device IO entry interface (generic, transport-agnostic)
       Contains only essential I/O methods for maximum flexibility.

Copyright (C) Atmosic 2025
'''

import abc
from typing import Optional


class DeviceIoEntryBase(abc.ABC):
    """Minimal device I/O interface with only essential methods."""

    @abc.abstractmethod
    def clear_write_buffer(self) -> None:
        """Check if device is open and ready for communication."""
        raise NotImplementedError

    @abc.abstractmethod
    def clear_read_buffer(self) -> None:
        """Check if device is open and ready for communication."""
        raise NotImplementedError

    @abc.abstractmethod
    def read(self, size: Optional[int] = None, timeout: Optional[float] = None
            ) -> Optional[bytes]:
        """Read data from the device.

        Args:
            size: Maximum number of bytes to read. None means read all available.
            timeout: Read timeout in seconds. None means use default.

        Returns:
            bytes if data available, None if no data or timeout.
        """
        raise NotImplementedError

    @abc.abstractmethod
    def write(self, data) -> int:
        """Write data to the device.

        Args:
            data: bytes or str to write

        Returns:
            Number of bytes written.
        """
        raise NotImplementedError

    @abc.abstractmethod
    def can_read(self) -> bool:
        """Check if device is ready for reading.

        Returns:
            True if device can be read from, False otherwise.
        """
        raise NotImplementedError

    @abc.abstractmethod
    def can_write(self) -> bool:
        """Check if device is ready for writing.

        Returns:
            True if device can be written to, False otherwise.
        """
        raise NotImplementedError

