
"""
Error Code System for Atmosic UART Loader Host

Error Code Format: VMMIII
- V: Version (1 digit) - Error code schema version
- MM: Major (2 digits) - Major error category (10-99)
- III: Minor (3 digits) - Specific error within category (001-999)

Examples:
- 123456: Version=1, Major=23, Minor=456
- 101001: Version=1, Major=10, Minor=001
- 102001: Version=1, Major=20, Minor=001
- 103001: Version=1, Major=30, Minor=001


Maximum expected error code: 999999
Unexpected/unknown errors: 1000000
"""
from __future__ import annotations
from enum import Enum

SUCCESS_EXIT_CODE = 0
MAX_EXPECTED_ERROR_CODE = 999999
UNEXPECTED_ERROR_CODE = MAX_EXPECTED_ERROR_CODE + 1
KEYBOARD_INTERRUPT_CODE = UNEXPECTED_ERROR_CODE + 1
RETRY_ERROR_CODE_BASE = MAX_EXPECTED_ERROR_CODE + 1000001
RETRY_SUCCESS = RETRY_ERROR_CODE_BASE + SUCCESS_EXIT_CODE

ERROR_CODE_VERSION = 1

MAX_MAJOR_ERROR_CODE = 99
MAX_MINOR_ERROR_CODE = 999

class MajorErrorCodes(int, Enum):
    """Major error codes for Atmosic applications"""
    HCI_STATUS_ERROR = 1
    SERIAL_IO_ENTRY_ERROR = 2
    DEVICE_IO_MODULE_ERROR = 3
    PACKET_MODULE_ERROR = 4
    HCI_MODULE_ERROR = 5
    HCI_GATEWAY_ERROR = 6
    UART_LOADER_AGENT_ERROR = 7
    HCI_COMMAND_OPERATION_ERROR = 8
    XMODEM_OPERATION_ERROR = 9

class HciStatusErrorCode(int, Enum):
    """Error codes for HCI status errors."""
    PROG_ERR_INVALID_PARAM = 0x01 # Invalid parameter
    PROG_ERR_CHECKSUM_FAIL = 0x02 # XOR checksum verification failed
    PROG_ERR_SEND_ERROR = 0x03 # Send data failed
    PROG_ERR_INSUFFICIENT_MEMORY = 0x04 # Insufficient memory
    PROG_ERR_XOR_MISMATCH = 0x05 # XOR verification mismatch
    PROG_ERR_SHA256_MISMATCH = 0x06 # SHA256 verification mismatch
    PROG_ERR_TIMEOUT = 0x07 # Operation timeout
    PROG_ERR_BUSY = 0x08 # System busy
    PROG_ERR_UNSUPPORTED_SIZE = 0x09 # Unsupported block size
    PROG_ERR_MEMORY_ACCESS = 0x0A # Memory access error
    PROG_ERR_PROTOCOL_VERSION = 0x0B # Protocol version error
    PROG_ERR_BAUDRATE_INVALID = 0x0C # Baudrate invalid

class SerialIoEntryErrorCode(int, Enum):
    """Error codes for serial I/O operations."""
    PORT_OPEN_FAILED = 1
    READ_FAILED = 2
    WRITE_FAILED = 3

class DeviceIoModuleErrorCode(int, Enum):
    """Error codes for DeviceIoModule operations."""
    READ_AUGUMENT_ERROR = 1
    READ_IMMEDIATE_NO_DATA = 2
    READ_TIMEOUT = 3

class PacketModuleErrorCode(int, Enum):
    """Error codes for PacketModule operations."""
    READ_UNTIL_TIMEOUT = 1
    WRITE_NOT_ENTIRELY = 2

class HciModuleErrorCode(int, Enum):
    """Error codes for HciModule operations."""
    READ_PACKET_TYPE_MISMATCH = 1

class HciGatewayErrorCode(int, Enum):
    """Error codes for HciGateway operations."""
    EVENT_FORMAT_ERROR = 1

class UartLoaderAgentErrorCode(int, Enum):
    """Error codes for UartLoaderAgent operations."""
    WAIT_BOOT_EVENT_TIMEOUT = 1
    PROG_WRITE_DATA_ARG_INVALID = 2
    PROG_WRITE_UNEXCPECTED_SEQ_ID = 3
    XOR_RAM_MISMATCH = 4
    XOR_STORAGE_MISMATCH = 5
    DUMP_TIMEOUT = 6
    DUMP_XOR_MISMATCH = 7

class UartLoaderTaskErrorCode(int, Enum):
    """Error codes for UartLoaderOperation operations."""
    DUMP_CHECK_SHA256_MISMATCH = 1
    PROG_FILE_NOT_EXIST = 2
    PROG_CHECK_SHA256_MISMATCH = 3
    PROG_ERASE_SIZE_SMALLER_THAN_BINARY_SIZE = 4

class XmodemOperationErrorCode(int, Enum):
    """Error codes for XmodemOperation operations."""
    XMODEM_UPLOAD_TIMEOUT = 1
    XMODEM_UPLOAD_FAILED = 2
    XMODEM_GETC_TIMEOUT = 3
    XMODEM_ASK_VERSION_TIMEOUT = 4
