"""
UART Loader Agent Data Types

Data structures returned by uart loader agent methods.
These replace direct HCI event returns with typed, business-focused structures.
"""

from dataclasses import dataclass
from typing import Optional


@dataclass
class DeviceInfo:
    """Device information structure.
    
    Returned by get_device_info() instead of raw HCI event.
    Contains parsed and validated device information.
    """
    app_version: int
    protocol_version: int
    ram_buffer_start: int
    ram_buffer_size: int
    num_flash: Optional[int] = None
    flash_ids: Optional[bytes] = None
    
    def __str__(self) -> str:
        """Human-readable device info representation."""
        return (f"DeviceInfo(app_ver=0x{self.app_version:04X}, "
                f"protocol_ver=0x{self.protocol_version:04X}, "
                f"ram_buffer=0x{self.ram_buffer_start:08X}+"
                f"0x{self.ram_buffer_size:08X})")

@dataclass
class ProgramResult:
    skiped_ff_size: int = 0
    skiped_ff_packets: int = 0
    skiped_ff_chunks: int = 0

@dataclass
class DiagnoTputResult:
    """Diagno throughput result structure.
    
    Returned by diagno_tput() instead of raw HCI event.
    Contains parsed and validated diagno throughput result.
    """
    write_time_ram: int
    erase_time_storage: int
    write_time_storage: int

    def __str__(self) -> str:
        """Human-readable diagno throughput result representation."""
        return (f"DiagnoTputResult(write_time_ram={self.write_time_ram}, "
                f"erase_time_storage={self.erase_time_storage}, "
                f"write_time_storage={self.write_time_storage})")